package com.jispot.common;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.TreeMap;

public abstract class PropertyFile {

	public Properties props;

    public PropertyFile() {}

    private String fileName = null;

    /**
     * Initialize the property file
     * 
     * @param propriete the property file (from classpth or environment)
     * @param fromClasspath true if the files comes from classpath, false otherwise
     */
    public void init( String propertyFile ) {
        this.fileName = propertyFile;

        try {
            InputStream is = this.getClass().getResourceAsStream( fileName );
            if ( is != null ) {
                if ( props == null ) {
                    props = new java.util.Properties();
                }
                props.load( is );
            }
            else {
                System.out.println( " InputStream null" );
            }
        }
        catch ( Throwable e ) {
            // Si une erreur survient ici, on est mal !!
            e.printStackTrace(); // To change body of catch statement use
            // Options | File Templates.
        }
    }

    public String getFileName() {
        return fileName;
    }

    /**
     * Retroune la valeur de la clef désignée
     * 
     * @param clef celle que l'on cherche
     * @return la valeur sous forme de Chaine de caractère
     */
    public String get( String clef ) {
        if ( props != null ) {
            return props.getProperty( clef );
        }
        return null;
    }

    /**
     * Retourne la valeur de la clef désignée
     * 
     * @param clef celle que l'on cherche
     * @return la valeur sous forme d'int, 0 si exception ! (oui je sais...)
     */
    public int getAsInt( String clef ) {
        if ( props != null ) {
            String v = get( clef );
            if ( v != null ) {
                try {
                    return Integer.parseInt( v );
                }
                catch ( Exception e ) {
                    return 0;
                }
            }
        }
        return 0;
    }

    /**
     * Retourne la valeur de la clef désignée
     * 
     * @param clef celle que l'on cherche
     * @return la valeur sous forme de long, 0 si exception ! (oui je sais...)
     */
    public long getAsLong( String clef ) {
        if ( props != null ) {
            String v = get( clef );
            if ( v != null ) {
                try {
                    return Long.parseLong( v );
                }
                catch ( Exception e ) {
                    return 0;
                }
            }
        }
        return 0;
    }

    /**
     * Liste de valeur sur les clefs en partant d'un element de la clef.<br>
     * Ex :<br>
     * this.getListe(<strong>key</strong>);<br>
     * key.TOTO = value1<br>
     * key.TITI = value2<br>
     * key.tata = value4<br>
     * resultat = ("TOTO", "TITI", "tata")<br>
     * 
     * @param clef a value of type 'String'
     * @return a value of type 'Collection'
     */
    public List<String> getListe( String clef ) {
        if ( props != null ) {
            List<String> al = new ArrayList<String>( props.size() );
            String s;
            int keySize = clef.length();
            Enumeration<?> enu = props.propertyNames();
            while ( enu.hasMoreElements() ) {
                s = ( String ) enu.nextElement();
                if ( s.startsWith( clef ) ) {
                    al.add( s.substring( keySize ) );
                }
            }
            return al;
        }
        return new ArrayList<String>();
    }

    /**
     * Liste de valeur.<br>
     * Ex :<br>
     * key.1 = value1<br>
     * key.2 = value2<br>
     * key.4 = value4<br>
     * resultat = ("value1", "value2")<br>
     * 
     * @param clef a value of type 'String'
     * @return a value of type 'Collection'
     */
    public List<String> getKeyName( String clef ) {
        if ( props != null ) {
            List<String> al = new ArrayList<String>( 20 );
            String s;
            int i = 1;
            while ( ( s = get( clef + i ) ) != null ) {
                al.add( s.trim() );
                i++;
            }
            return al;
        }
        return new ArrayList<String>();
    }

    /**
     * Liste de valeur.<br>
     * Ex :<br>
     * key.toto = value1,value2,value4<br>
     * resultat = ("value1", "value2", "value4")<br>
     * 
     * @param key a value of type 'String'
     * @return a value of type 'List'
     */
    public List<String> getListeSimple( String key ) {
        return getListeSimple( key , "," );
    }

    /**
     * Liste de valeur.<br>
     * Ex :<br>
     * key.toto = value1,value2,value4<br>
     * resultat = ("value1", "value2", "value4")<br>
     * 
     * @param key a value of type 'String'
     * @param delim the delimiter we want to use
     * @return a value of type 'List'
     */
    public List<String> getListeSimple( String key , String delim ) {
        if ( props != null && get( key ) != null ) {
            StringTokenizer tok = new StringTokenizer( get( key ) , delim );
            List<String> liste = new ArrayList<String>( tok.countTokens() );
            while ( tok.hasMoreElements() ) {
                String chaine = ( String ) tok.nextElement();
                liste.add( chaine.trim() );
            }
            return liste;
        }
        return new ArrayList<String>();
    }

    /**
     * Returns a Map of values. Ex :<br>
     * key.1=value1<br>
     * key.2=value2<br>
     * key.4=value4<br>
     * Result = ("1"="value1", "2"="value2", "4"="value4")
     * 
     * @param clef a value of type 'String'
     * @return a value of type 'Collection'
     */
    public Map<String , String> getMap( final String clef ) {
        if ( props != null ) {
            int keySize = clef.length();
            Map<String , String> m = new TreeMap<String , String>();
            String s;
            String key;
            Enumeration<?> e = props.propertyNames();
            while ( e.hasMoreElements() ) {
                s = ( String ) e.nextElement();
                if ( s.startsWith( clef ) ) {
                    key = s.substring( keySize );
                    m.put( key.trim() , get( s ) );
                }
            }
            return m;
        }
        return new HashMap<String , String>();
    }

    /**
     * Est capable de gérer TRUE / YES / ON / 1 Tout le reste est false...
     * 
     * @param propertie la clef de la propriété
     * @return true si TRUE / YES / ON / 1
     */
    public boolean getAsBoolean( String propertie ) {
        if ( props != null ) {
            String val = props.getProperty( propertie );
            if ( "true".equalsIgnoreCase( val ) || "yes".equalsIgnoreCase( val ) || "on".equalsIgnoreCase( val )
                    || "1".equalsIgnoreCase( val ) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Retourne une liste de clefs contenant la chaîne de caractères <BR>
     * passée en paarmètre
     * 
     * @param sentance la chaîne de caractères à rechercher
     * @return la liste de clefs contenant la chaîne de caractères
     */
    public List<String> getKeyListFromSentance( String sentance ) {
        if ( props != null ) {
            List<String> keyList = new ArrayList<String>( props.size() );
            String key;
            Enumeration<?> enu = props.keys();

            while ( enu.hasMoreElements() ) {
                key = ( String ) enu.nextElement();
                if ( key.contains( sentance ) ) {
                    keyList.add( key );
                }
            }
            return keyList;
        }
        return new ArrayList<String>();
    }

}
