package com.jispot.common;

import java.util.logging.Logger;

import com.jispot.common.i18n.I18nError;

/**
 * The Class JISpotEx represents a Throwable and simplifies the use of ERROR CODES.<br>
 * Standart ERROR CODES can be found in JISpotExCommon and JISpotExAny.<br>
 * <br>
 * <strong>Simpliest usage is:</strong> <code>throw new JISpotEx(JISpotCommonEx.C5);<code>
 * or:
 * <code>throw new JISpotEx(JISpotAnyEx.A822);</code>
 * 
 * It will throw an exception containing an error code.<br>
 * <br>
 * <strong>If you want to traduce the error into a specific language:</strong><br>
 * <code>throw new JISpotEx(JISpotCommonEx.C5 , userContext.getLangage() );</code> or:<br>
 * <code>throw new JISpotEx(JISpotAnyEx.A822 , Default2MProperties.getLangage() );<code><br>
 * <br>
 * <strong>Language (or not) + a complementary message:</strong><br>
 * <code>throw new JISpotEx(JISpotCommonEx.C5 , userContext.getLangage() , "error_message_my_very_specific_message" );</code>
 * or:<br>
 * <code>throw new JISpotEx(JISpotAnyEx.A822 , null, "error_message_my_very_specific_message" );</code><br>
 * <br>
 * <strong>Language (or not) and a complementary message that takes arguments:</strong><br>
 * <code>throw new JISpotEx(JISpotCommonEx.C5 , null , "error_message_group_missing", new String[] {"Operators"} );<code>
 * or:<br>
 * <code>throw new JISpotEx(JISpotCommonEx.C12 , userContext.getLangage() , "error_message_arguments_wrong", new String[] {"28" , "Username"} );</code>
 */
public class JISpotEx extends Throwable {

	protected final Logger log = Logger.getLogger(this.getClass().getName());

    /** The JISpot Error CODE. */
    public int              code;

    /** The JISpot Error CODE description traduced. */
    public String           codeDesc;

    /** The detail (optionnaly parsed with params) traduced. */
    public String           detail;

    /**
     * Simple declaration of an error.<br>
     * 
     * @param code the JISpot error code
     */
    public JISpotEx( int code ) {
        super( ( String ) null );
        this.code = code;
        this.codeDesc = I18nError.get().getString( "jispot_error_code_" + code );
    }

    /**
     * Simple declaration of an error with an original exception.<br>
     * 
     * @param code the JISpot error code
     * @param exception the original exception
     */
    public JISpotEx( int code , Throwable exception ) {
        super( exception );
        this.code = code;
        this.codeDesc = I18nError.get().getString( "jispot_error_code_" + code );
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     */
    public JISpotEx( int code , String locale ) {
        super( ( String ) null );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString( "jispot_error_code_" + code );
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     * @param exception the original exception
     */
    public JISpotEx( int code , String locale , Throwable exception ) {
        super( exception );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString( "jispot_error_code_" + code );
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     * @param detail an i18n key that could explain a little more the error
     */
    public JISpotEx( int code , String locale , String detail ) {
        super( ( String ) null );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString("jispot_error_code_" + code );
        this.detail = I18nError.get(locale).getString( detail );
        if(this.codeDesc == null || this.detail == null ){
            this.detail = detail;
        }
    
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     * @param detail an i18n key that could explain a little more the error
     * @param exception the original exception
     */
    public JISpotEx( int code , String locale , String detail , Throwable exception ) {
        super( exception );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString("jispot_error_code_" + code );
        this.detail = I18nError.get(locale).getString( detail );
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     * @param detail an i18n key that could explain a little more the error
     * @param params if your <strong>detail</strong> needs parameters
     */
    public JISpotEx( int code , String locale , String detail , String[] params ) {
        super( ( String ) null );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString("jispot_error_code_" + code );
        this.detail = StringUtils.replaceParams( I18nError.get(locale).getString(detail) , params );
    }

    /**
     * Instantiates a new JISpot ex.
     * 
     * @param code the JISpot error code
     * @param locale the locale to use for translation
     * @param detail an i18n key that could explain a little more the error
     * @param params if your <strong>detail</strong> needs parameters
     * @param exception the original exception
     */
    public JISpotEx( int code , String locale , String detail , String[] params , Exception exception ) {
        super( exception );
        this.code = code;
        this.codeDesc = I18nError.get(locale).getString("jispot_error_code_" + code );
        this.detail = StringUtils.replaceParams( I18nError.get(locale).getString(detail) , params );
    }

    /**
     * This log method outputs the ERROR code and appends the description into the user's
     * language.
     * <br>
     * ERROR CODE XX: CodeDesc<br>
     * -----> Detail [with params]<br>
     * Cause: Original Exception reason
     */
    public void log() {
        String out = "ERROR " + code + ": " + codeDesc;
        // If there is a specific detail with arguments let's add it
        if ( !StringUtils.isNullOrEmpty( detail ) ) {
            out += "\n-----> " + detail;
        }
        log.warning( out );

        if ( !StringUtils.isNullOrEmpty( super.getMessage() ) ) {
            log.finest( "Cause: " + super.getMessage() );
        }
    }

}
